package furny.ga.util;

import java.util.List;

import com.jme3.math.FastMath;

import furny.entities.Furniture;
import furny.furndb.FurnCache;
import furny.ga.FurnEntry;
import furny.ga.FurnEntryList;
import furny.ga.FurnGA;
import furny.ga.FurnLayoutIndividual;
import furny.ga.PseudoSpace;
import furny.ga.RoomVector;
import ga.core.GA;
import ga.core.validation.GAContext;
import ga.view.interfaces.IPhenotypeSpace;

/**
 * Utility class for calculating furniture layout individual distance. The
 * distance is always normalized (in [0;1]).
 * 
 * @since 11.08.2012
 * @author Stephan Dreyer
 */
public final class Distance {

  /**
   * Instantiation is not allowed.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  private Distance() {
  }

  /**
   * Calculates the distance of two chromosomes in the interval of [0;1].
   * 
   * @param ind1
   *          The first individual
   * @param ind2
   *          The second individual
   * @param context
   *          The GA context
   * @deprecated This old distance calculation was not working well.
   * @return The normalized distance of the two individuals
   * 
   * @since 18.02.2012
   * @author Stephan Dreyer
   */
  @Deprecated
  public static float calcDistance(final FurnLayoutIndividual ind1,
      final FurnLayoutIndividual ind2, final GAContext context) {
    final FurnEntryList list1 = ind1.getFurnitures();
    final FurnEntryList list2 = ind2.getFurnitures();

    float dist = 0f;

    for (final FurnEntry e1 : list1) {
      for (final FurnEntry e2 : list2) {

        final double d = furnEntryDistance(e1, e2, context);
        // System.err.println("dist: " + d);
        dist += d;
      }
    }

    // TODO crappy distance calculation

    // normalize to the number of furniture pairs
    return dist / (list1.size() * list2.size());
  }

  /**
   * Distance calculation for two furniture layouts.
   * 
   * @param ind1
   *          The first individual.
   * @param ind2
   *          The second individual.
   * @param context
   *          The GA context.
   * @return Distance in [0;1].
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public static float calcDistance2(final FurnLayoutIndividual ind1,
      final FurnLayoutIndividual ind2, final GAContext context) {
    final FurnEntryList list1 = ind1.getFurnitures();
    final FurnEntryList list2 = ind2.getFurnitures();

    final float d1 = leftFurnEntryDistance(list1, list2, context);
    final float d2 = leftFurnEntryDistance(list2, list1, context);

    // normalize to the number of furniture pairs
    final float dist = (d1 + d2) / 2f;

    return dist;
  }

  /**
   * Left distance calculation of two furniture entry lists.
   * 
   * @param l1
   *          The left list.
   * @param l2
   *          The right list.
   * @param context
   *          The GA context.
   * @return The distance in [0;1].
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  private static float leftFurnEntryDistance(final FurnEntryList l1,
      final FurnEntryList l2, final GAContext context) {
    if (l1.isEmpty() && l2.isEmpty()) {
      return 0f;
    } else if (l1.isEmpty() != l2.isEmpty()) {
      return 1f;
    }

    float dist = 0f;

    for (final FurnEntry fe : l1) {
      final FurnEntryList list = l2.getEntries(fe.getFurniture());
      if (!list.isEmpty()) {
        dist += lowestFurnEntryDistance(fe, list, context);
      } else {
        dist += lowestFurnEntryDistance(fe, l2, context);
      }
    }

    return dist / l1.size();
  }

  /**
   * Calculates the lowest distance between a furniture entry and a list of
   * other furniture entries.
   * 
   * @param entry
   *          The furniture entry.
   * @param others
   *          The other entries.
   * @param context
   *          The GA context.
   * @return The distance in [0;1].
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  private static float lowestFurnEntryDistance(final FurnEntry entry,
      final FurnEntryList others, final GAContext context) {
    float dist = 1f;

    for (final FurnEntry fe : others) {
      final float dTemp = furnEntryDistance(entry, fe, context);

      if (dTemp < dist) {
        dist = dTemp;
      }
    }

    return dist;
  }

  /**
   * Calculates the distance of two genes (FurnEntrys) in the interval of [0;1].
   * 
   * @param e1
   *          The first FurnEntry
   * @param e2
   *          The second FurnEntry
   * @param context
   *          The ga context
   * @return The normalized distance between the two genes.
   * 
   * @since 14.02.2012
   * @author Stephan Dreyer
   */
  private static float furnEntryDistance(final FurnEntry e1,
      final FurnEntry e2, final GAContext context) {
    final Object o = context.get(GA.KEY_VALIDATION_SPACE);

    if (o != null && o instanceof IPhenotypeSpace) {
      final float wID = context.getFloat(FurnGA.KEY_WEIGHT_ID, 8f);
      final float wTrans = context.getFloat(FurnGA.KEY_WEIGHT_TRANSLATION, 2f);
      final float wRot = context.getFloat(FurnGA.KEY_WEIGHT_ROTATION, 1f);

      float dist = 0f;

      final IPhenotypeSpace space = (IPhenotypeSpace) o;

      final float roomWidth = (float) space.getOutterBounds().getWidth();
      final float roomLength = (float) space.getOutterBounds().getHeight();

      // euklidian distance in 2d, normalized to the maximal size of the room
      dist += (e1.getVector().distance(e2.getVector()) / FastMath.sqrt(FastMath
          .pow(roomWidth, 2f) + FastMath.pow(roomLength, 2f)))
          * wTrans;

      // rotational distance, normalized
      dist += (FastMath.abs(e1.getVector().diffRotation(e2.getVector())) / 2d)
          * wRot;

      if (!e1.getFurniture().equals(e2.getFurniture())) {
        // normalized similarity between furnitures
        final double sim = FurnitureUtil.getSimilarity(e1.getFurniture(),
            e2.getFurniture());

        // distance = 1-similarity
        dist += (1d - sim) * wID;
      }

      // normalize to sum of the weights
      return dist / (wID + wTrans + wRot);

    } else {
      throw new RuntimeException("Phenotype space not found in the ga context");
    }
  }

  /**
   * Main method for testing.
   * 
   * @param args
   *          No arguments
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public static void main(final String[] args) {
    final GAContext context = new GAContext();
    context.put(GA.KEY_VALIDATION_SPACE, new PseudoSpace(10f, 10f));
    context.put(FurnGA.KEY_WEIGHT_ID, 8);
    context.put(FurnGA.KEY_WEIGHT_TRANSLATION, 2);
    context.put(FurnGA.KEY_WEIGHT_ROTATION, 1);

    final FurnLayoutIndividual ind1 = new FurnLayoutIndividual(context);
    final List<Furniture> all = FurnCache.getInstance().getAllFurnitures();

    ind1.getFurnitures().add(
        new FurnEntry(new RoomVector(-500, -500, 1), all.get(1)));

    ind1.getFurnitures().add(
        new FurnEntry(new RoomVector(50, 50, 1), all.get(0)));

    final FurnLayoutIndividual ind2 = new FurnLayoutIndividual(context);
    ind2.getFurnitures().add(
        new FurnEntry(new RoomVector(500, 500, 3), all.get(1)));

    ind2.getFurnitures().add(
        new FurnEntry(new RoomVector(100, 150, 2), all.get(1)));

    ind2.getFurnitures().add(
        new FurnEntry(new RoomVector(100, 100, 2), all.get(0)));

    System.err.println(ind1);

    System.err.println(ind2);

    System.err.println("Failers");
    System.err.println(calcDistance(ind1, ind2, context));
    System.err.println(calcDistance(ind2, ind1, context));
    System.err.println(calcDistance(ind1, ind1, context));
    System.err.println();
    System.err.println("New calculation");
    System.err.println(calcDistance2(ind1, ind2, context));
    System.err.println(calcDistance2(ind2, ind1, context));
    System.err.println(calcDistance2(ind1, ind1, context));
  }
}
